using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents response message to successful logon onto M server.
	/// </summary>
	public sealed class SecurityLogonSuccessResponseMessage : SecurityBaseResponseMessage
	{
		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secLogonResponse.xsd";

		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "AV.Logon";

		private string _postSignInText;

		/// <summary>
		/// Constructor that needs only a post sign-in text to create a message.
		/// </summary>
		/// <param name="postSignInText">Server post sign-in text message. Required.</param>
		public SecurityLogonSuccessResponseMessage( string postSignInText )
			: base( XsdName, ResponseType, ResponseStatus.Success ) 
		{
			if( postSignInText == null )
				throw( new ArgumentNullException( "postSignInText" ) );

			_postSignInText = postSignInText;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityLogonSuccessResponseMessage( XmlDocument sourceDoc )
			: base( XsdName, ResponseType, sourceDoc )
		{
			if( _responseStatus != ResponseStatus.Success ) 
				throw( new XmlParseException( SR.Exceptions.SecLogonSuccessRespMsgInvalidStatus( _responseStatus.ToString() ) ) );

			_postSignInText = 
				SecurityLogonNonFailureResponseMessageImplementationMixin.ParsePostSignInTextXml( XmlParseGetResponseElement( sourceDoc ), true );
		}

		/// <summary>
		/// Overriden method adding message-specific content to the response element
		/// created by the base class. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			SecurityLogonNonFailureResponseMessageImplementationMixin.WritePostSignInTextToXml( writer, _postSignInText, true );

			base.WriteXmlResponseNodeContent( writer );
		}

		/// <summary>
		/// Server post sign-in text that should be shown to user. 
		/// </summary>
		public string PostSignInText
		{
			get
			{
				return _postSignInText;
			}
		}
	}
}
